import { Dimension, GameMode, ItemStack, Player, RawMessage, VanillaEntityIdentifier, Vector3 } from "@minecraft/server";

type ActionType =
    |{type: "form"; value: string}
    |{type: "sound"; value: string|{soundId: string; volume?: number; pitch?: number; target?: "all"|"self"}}
    |{type: "say"; value: RawMessage|string}
    |{type: "give"; value: {itemId: ItemStack|string; amount: number}}
    |{type: "cmds"; value: string[]}
    |{type: "eval"; value: string}
    |{type: "tellraw"; value: RawMessage|string}
    |{type: "add_tag"; value: RawMessage|string}
    |{type: "remove_tag"; value: RawMessage|string}
    |{type: "gamemode"; value: GameMode|{selector: Player; gamemode: GameMode}}
    |{type: "tp"|"teleport"; value: {location: Vector3|{x: number, y: number, z: number}; dimension: Dimension}}
    |{type: "summon"; value: VanillaEntityIdentifier|{location: Player|Vector3; entity: VanillaEntityIdentifier}}

type ModalActionType =
    |{type: "give"; value: {selector: "selector"|"self"|"all"; itemId: "itemId"; amount: "amount"}}
    |{type: "add_tag"; value: {selector: "selector"|"self"|"all"; tag: "tag"}}
    |{type: "remove_tag"; value: {selector: "selector"|"self"|"all"; tag: "tag"}}
    |{type: "effect"; value: {selector: "selector"|"self"|"all"; effects: "effects"; amplifier?: "amplifier"; duration?: "duration"; particles?: "particles"}}
    |{type: "scoreboard"; value: {selector: "selector"|"self"|"all"; objective: "objective", amount: "amount"}}
    |{type: "message"; value: {selector: "selector"|"self"|"all"; message: "message"; colour?: "colour"; format?: "format"}}
    |{type: "confirm"; value: {selector?: "selector"|"self"|"all"; itemId?: "itemId"; amount?: "amount"; tag?: "tag"; effects?: "effects"; amplifier?: "amplifier"; duration?: "duration"; particles?: "particles"; message?: "message"; colour?: "colour"; format?: "format"; gamemode?: "gamemode"}; body: RawMessage|string; cancelText: RawMessage|string; confirmText: RawMessage|string; cancelAction: ConfirmActionType[]; confirmAction: ConfirmActionType[]}
    |{type: "gamemode"; value: {selector: "selector"|"self"|"all"; gamemode: "gamemode"}}
    |{type: "tp"|"teleport"; value: {selector: "selector"|"self"|"all"; destination: Vector3|"destination"}}
    |{type: "summon"; value: {selector?: "selector"|"self"|"all"; location?: Vector3|"location"; entity: "entity"|VanillaEntityIdentifier}};

type ConfirmActionType =
    |{type: "form"; value: string}
    |{type: "sound"; value: string|{soundId: string; volume?: number; pitch?: number; target?: "{#SELF}"|"{#ALL}"|"all"|"self"}}
    |{type: "give"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; itemId: "{#ITEMID}"; amount: "{#AMOUNT}"}}
    |{type: "add_tag"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; tag: "{#TAG}"}}
    |{type: "remove_tag"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; tag: "{#TAG}"}}
    |{type: "effect"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; effects: "{#EFFECTS}"; amplifier?: "{#AMPLIFIER}"; duration?: "{#DURATION}"; particles?: "{#PARTICLES}"}}
    |{type: "scoreboard"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; objective: "{#OBJECTIVE}", amount: "{#AMOUNT}"}}
    |{type: "message"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; message: "{#MESSAGE}"; colour?: "{#COLOUR}"; format?: "{#FORMAT}"}}
    |{type: "gamemode"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; gamemode: "{#GAMEMODE}"}}
    |{type: "tp"|"teleport"; value: {selector: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; destination: "{#DESTINATION}"}}
    |{type: "summon"; value: {selector?: "{#SELECTOR}"|"{#SELF}"|"{#ALL}"|"self"|"all"; location?: "{#LOCATION}"|Vector3; entity: "{#Entity}"|VanillaEntityIdentifier}};

type TellrawPart =
    |{text: RawMessage|string}
    |{selector: string}
    |{translate: RawMessage|string};

interface BaseForm {
    /**
     * @remarks
     * This builder method sets the title for the modal dialog.
     */
    title: RawMessage|string;
    /**
     * @remarks`    
     * This is the event(s) that triggers when the form opens
     */
    onOpen?: ActionType[];
    /**
     * @remarks
     * This is the event(s) that triggers when the form opens
     */
    onClose?: ActionType[];
}

interface ActionForm extends BaseForm {
    type: "action";
    /**
     * @remarks
     * Method that sets the body text for the modal form.
     */
    body: RawMessage|string;
    /**
     * @remarks
     * Adds buttons to this form with an icon from a resource
     * pack. Can also do labels, dividers, and headers
     */
    elements: ActionElement[];
}

interface ModalForm extends BaseForm {
    type: "modal";
    /**
     * The list of elements
     */
    elements: FormElement[];
}

interface MessageForm extends BaseForm {
    type: "message";
    /**
     * @remarks
     * Method that sets the body text for the modal form.
     */
    body: RawMessage|string;
    /**
     * @remarks
     * Method that sets the text for the first and second
     * button of the dialog.
     */
    elements: MessageElement[];
}

type ActionElement =
    |{
        type: "button";
        /**
         * @remarks 
         * The name that the button will show.
         */
        buttonName: RawMessage|string;
        /**
         * @remarks
         * The optional icon that can be set from a texture
         * file path.
         */
        icon?: TexturePath;
        /**
         * @remarks
         * Defines the Event that will trigger when the button
         * is clicked.
         */
        onClick?: ActionType[];
    }
    |{type: "divider"}
    |{type: "header"; headerText: RawMessage|string}
    |{type: "label"; labelText: RawMessage|string};

// Update/Create actions to work with the specific Elements
type FormElement =
    // Default Slider
    |{type: "slider"; label: RawMessage|string; range: {min: number; max: number; step?: number, default?: number}; tooltip?: RawMessage|string; value: RawMessage|string}
    // Dropdown with players
    |{type: "player"; label: RawMessage|string; tooltip?: RawMessage|string; value: "selector"}
    // Default Dropdown
    |{type: "dropdown"; label: RawMessage|string; items: string[]; default?: number; tooltip?: RawMessage|string; value: RawMessage|string}
    // Default Toggle
    |{type: "toggle"; label: RawMessage|string; default?: boolean; tooltip?: RawMessage|string; value: RawMessage|string}
    // Default Text Field
    |{type: "textField"; label: RawMessage|string; placeholder: RawMessage|string; tooltip?: RawMessage|string; value: RawMessage|string; default?: string}
    // Default Submit Button
    |{type: "submitButton"; label?: RawMessage|string; onSubmit: ModalActionType[]}
    // A Divider
    |{type: "divider"}
    |{type: "header"; label: RawMessage|string}
    |{type: "label"; label: RawMessage|string};

// Message Forms
type MessageElement =
    |{
        type: "button1";
        /**
         * Recommended to have "No/Cancel" option for "button1"
         */
        label: RawMessage|string;
        onClick?: ActionType[];
    }|{
        type: "button2";
        /**
         * Recommended to have "Yes/OK" option on "button2"
         */
        label: RawMessage|string;
        onClick?: ActionType[];
    };


type FormDefinition = ActionForm|ModalForm|MessageForm;

type TexturePath = `textures/${string}`;

export type FormList = Record<string, FormDefinition>;